#ifndef CE2_RSA_SCHEMES_H
#define CE2_RSA_SCHEMES_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
 *  Object %CE2_RSA_SCHEMES.h    : %
 *  State           :  %state%
 *  Creation date   :  %date%
 *  Last modified   :  %modify_time%
 */
/** @file
 *  \brief This module defines the API that supports PKCS#1 v1.5 and PKCS#1 v2.1 schemes
 *
 *  \version CE2_SCHEMES.h#1:incl:1
 *  \author ronys ohads
 *  \remarks Copyright (C) 2001-2005 by Discretix Technologies Ltd. All Rights reserved.
 */

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  CE2_DX_RSA_SCHEMES_Encrypt
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key data structure of the User;
*  @param hashFunc       [in] - The hash function to be used;
*         Currently only CE2_PKCS1_SHA1 is supported. 
*         The other hash functions recommended 
*         by PKCS#1 v2.1 are SHA-256/284/512.
*  @param L              [in] - The label input;
*         Version 2.1 of the standard defines L as the empty string, 
*         so this argument is currently ignored. 
*  @param Llen           [in] - The label length;
*         Version 2.1 of the standard defines L as the empty string, 
*         so this argument is currently ignored.
*  @param MGF            [in] - The mask generation function. PKCS#1 v2.1 defines MGF1, 
*         so the only value allowed here is CE2_PKCS1_MGF1;
*  @param DataIn_ptr     [in] - A pointer to the data to encrypt;
*  @param DataInSize       [in] - The size, in bytes, of the data to encrypt;
*         \note Must be <= (modulus_size - 2*Hash_output_length - 2).
*  @param Output_ptr     [out] - A pointer to the encrypted data;
*         The buffer must be at least PubKey_ptr->N.len bytes long 
*         (that is, the size of the modulus, in bytes).
*  @param PKCS1_ver      [in] - Ver 1.5 or 2.1, according to the functionality required.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  This function implements the Encrypt algorithm as defined in
*  PKCS#1 v2.1 and PKCS#1 v1.5.
*  The actual macro that will be used by the user is:
*   CE2_RSA_OAEP_Encrypt 		- for v2.1;
*   CE2_RSA_PKCS1v15_Encrypt	- for v1.5.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DX_RSA_SCHEMES_Encrypt
*     to encrypt data using public key, hash function and mask
*     generation functuion with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_RSA_SCHEMES_Encrypt(CE2_RSAUserPubKey_t *UserPubKey_ptr,
                                    CE2_RSA_HASH_OpMode_t hashFunc,
                                    DxUint8_t *L,
                                    DxUint16_t Llen,
                                    CE2_PKCS1_MGF_t MGF,
                                    DxUint8_t     *DataIn_ptr,
                                    DxUint16_t     DataInSize,
                                    DxUint8_t     *Output_ptr,
                                    CE2_PKCS1_version PKCS1_ver);

/**
****************************************************************
* Function Name: 
*  CE2_RSA_OAEP_Encrypt
*
* Inputs:
*  see CE2_DX_RSA_SCHEMES_Encrypt
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_OAEP_Encrypt implements the RSAES-OAEP algorithm 
*  as defined in PKCS#1 v2.1 8.1
*  \note It is not recommended to use hash MD5 in OAEP PKCS1 ver 2.1, therfore 
*  it is not supported and the hash function argument has been eliminated
*  This function combines the RSA encryption primitive and the
*  EME-OAEP encoding method, to provide an RSA-based encryption
*  method that is semantically secure against adaptive
*  chosen-ciphertext attacks. For additional details, please refer to
*  the PKCS#1 standard.
*  \note The \em L input is defined by the standard as the empty
*  string, therefore the caller need not provide it.
*
* \b
* Algorithm:
*  -# Call CE2_DX_RSA_SCHEMES_Encrypt.
***************************************************************/
#define CE2_RSA_OAEP_Encrypt(UserPubKey_ptr,HashMode,L,Llen,MGF,Data_ptr,DataSize,Output_ptr)\
		CE2_DX_RSA_SCHEMES_Encrypt(UserPubKey_ptr,HashMode/*CE2_RSA_HASH_SHA1_mode*/,L,Llen,MGF,Data_ptr,DataSize,Output_ptr,CE2_PKCS1_VER21)

/**
****************************************************************
* Function Name: 
*  CE2_RSA_PKCS1v15_Encrypt
*
* Inputs:
*  see CE2_DX_RSA_SCHEMES_Encrypt
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_PKCS1v15_Encrypt implements the RSAES-PKCS1v15 algorithm 
   as defined in PKCS#1 v2.1 8.2
*
* \b
* Algorithm:
*  -# Call CE2_DX_RSA_SCHEMES_Encrypt.
***************************************************************/
#define CE2_RSA_PKCS1v15_Encrypt(UserPubKey_ptr,DataIn_ptr,DataInSize,Output_ptr)\
		CE2_DX_RSA_SCHEMES_Encrypt(UserPubKey_ptr,CE2_RSA_HASH_NO_HASH_mode,DX_NULL,0,CE2_PKCS1_NO_MGF,DataIn_ptr,DataInSize,Output_ptr,CE2_PKCS1_VER15)

/**
****************************************************************
* Function Name: 
*  CE2_DX_RSA_SCHEMES_Decrypt
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key data structure;
*         \note The representation (pair or quintuple) 
*         and hence the algorithm (CRT or not CRT) 
*         is determined by the Private Key data structure - 
*         using CE2_Build_PrivKey or CE2_Build_PrivKeyCRT 
*         determines which algorithm will be used.
*  @param hashFunc        [in] - The hash function to be used;
*         Currently only CE2_PKCS1_SHA1 is supported. 
*         The other hash functions recommended by 
*         PKCS#1 v2.1 are SHA-256/284/512.
*  @param L               [in] - The label input;
*         Version 2.1 of the standard defines L as the empty string, 
*         so this argument is currently ignored. 
*  @param Llen            [in] - The label length;
*         Version 2.1 of the standard defines L as the empty string, 
*         so this argument is currently ignored. 
*  @param MGF             [in] - The mask generation function. PKCS#1 v2.1 defines MGF1, 
          so the only value allowed here is CE2_PKCS1_MGF1.
*  @param Data_ptr        [in] - A pointer to the data to decrypt;
*  @param DataSize        [in] - The size, in bytes, of the data to decrypt;
*         \note Must be <= the size of the modulus.
*  @param Output_ptr      [out] - Pointer to the decrypted data;
*         The buffer must be at least PrivKey_ptr->N.len bytes long 
*         (that is, the size of the modulus, in bytes).
*  @param OutputSize_ptr  [in/out] - The size of the Output_buffer ptr  [bytes];
*         The input size value must be at least the size of the modulus 
*         This value is updated with the actual number 
*         of bytes that are loaded to Output_ptr buffer
*  @param PKCS1_ver       [in] - Ver 1.5 or 2.1, according to the functionality required.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  This function implements the Decrypt algorithm as defined
*  in PKCS#1 v2.1 and PKCS#1 v1.5.
*   The actual macro that will be used by the user is:
*   CE2_RSA_OAEP_Decrypt		- for v2.1;
*   CE2_RSA_PKCS1v15_Decrypt	- for v1.5.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DX_RSA_SCHEMES_Decrypt
*     to decrypt data using private key, hash function and mask
*     generation functuion with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_RSA_SCHEMES_Decrypt(CE2_RSAUserPrivKey_t *UserPrivKey_ptr,
                                    CE2_RSA_HASH_OpMode_t hashFunc,
                                    DxUint8_t *L,
                                    DxUint16_t Llen,
                                    CE2_PKCS1_MGF_t MGF,
                                    DxUint8_t     *Data_ptr,
                                    DxUint16_t     DataSize,
                                    DxUint8_t     *Output_ptr,
                                    DxUint16_t    *OutputSize_ptr,
                                    CE2_PKCS1_version PKCS1_ver);

/**
****************************************************************
* Function Name: 
*  CE2_RSA_OAEP_Decrypt
*
* Inputs:
*  see CE2_DX_RSA_SCHEMES_Decrypt
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  CE2_RSA_OAEP_Decrypt implements the RSAES-OAEP algorithm 
*  as defined in PKCS#1 v2.1 8.1
*  Note: It is not recommended to use hash MD5 in OAEP PKCS1 ver 2.1, therefore 
*        it is not supported and the hash function argument is eliminated
*  
*  This function combines the RSA decryption primitive and the
*  EME-OAEP encoding method, to provide an RSA-based decryption
*  method that is semantically secure against adaptive
*  chosen-ciphertext attacks. For more details, please refer to
*  the PKCS#1 standard.
*  \note The \em L input is defined by the standard as the empty
*  string, therefore the caller need not provide it.   
*
* \b
* Algorithm:
*  -# call CE2_DX_RSA_SCHEMES_Decrypt
***************************************************************/
#define CE2_RSA_OAEP_Decrypt(UserPrivKey_ptr,HashMode,L,Llen,MGF,Data_ptr,DataSize,Output_ptr,OutputSize_ptr)\
		CE2_DX_RSA_SCHEMES_Decrypt(UserPrivKey_ptr,HashMode/*CE2_RSA_HASH_SHA1_mode*/,L,Llen,MGF,Data_ptr,DataSize,Output_ptr,OutputSize_ptr,CE2_PKCS1_VER21)

/**
****************************************************************
* Function Name: 
*  CE2_RSA_PKCS1v15_Decrypt
*
* Inputs:
*  see CE2_DX_RSA_SCHEMES_Decrypt
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  CE2_RSA_PKCS1v15_Decrypt implements the RSAES-PKCS1v15 algorithm as defined
*  in PKCS#1 v2.1 8.2
*
* \b
* Algorithm:
*  -# call CE2_DX_RSA_SCHEMES_Decrypt
***************************************************************/
#define CE2_RSA_PKCS1v15_Decrypt(UserPrivKey_ptr,DataIn_ptr,DataInSize,Output_ptr,OutputSize_ptr)\
		CE2_DX_RSA_SCHEMES_Decrypt(UserPrivKey_ptr,CE2_RSA_HASH_NO_HASH_mode,DX_NULL,0,CE2_PKCS1_NO_MGF,DataIn_ptr,DataInSize,Output_ptr,OutputSize_ptr,CE2_PKCS1_VER15)

/**
****************************************************************
* Function Name: 
*  CE2_DX_RSA_Sign
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key data
*         structure of the user;
*         \note The representation
*         (pair or quintuple) and hence the
*         algorithm (CRT or not) is determined
*         by the Private Key data
*         structure - using CE2_Build_PrivKey
*         or CE2_Build_PrivKeyCRT determines
*         which algorithm will be used.
*  @param hashFunc        [in] - The hash function to be used;
*         Currently only CE2_PKCS1_SHA1 and CE2_PKCS1_MD5 are supported.
*  @param MGF             [in] - The mask generation function;
*         Only for PKCS#1 v2.1 defines MGF1,
*         so the only value allowed for v2.1 is CE2_PKCS1_MGF1.
*  @param SaltLen         [in] - The Length of the Salt buffer;
*         Typical lengths are 0 and hLen (20 for SHA1)
*         The maximum length allowed is NSize - hLen - 2
*  @param DataIn_ptr      [in] - A pointer to the data to sign;
*  @param DataInSize      [in] - The size, in bytes, of the data to sign;
*  @param Output_ptr      [out] - A pointer to the signature;
*         The buffer must be at least PrivKey_ptr->N.len bytes long 
*         (that is, the size of the modulus in bytes).
*  @param OutputSize_ptr  [in/out] - A pointer to the Signature Size value -
*         the input value is the signature buffer size allocated, the output value is 
*         the signature size actually used.
*         The buffer must be at least PrivKey_ptr->N.len bytes long 
*         (that is, the size of the modulus in bytes).
*  @param PKCS1_ver       [in] - Ver 1.5 or 2.1, according to the functionality required.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_RSA_INVALID_PRIV_KEY_STRUCT_POINTER_ERROR,
*     - CE2_RSA_PRIV_KEY_VALIDATION_TAG_ERROR,
*     - CE2_RSA_HASH_ILLEGAL_OPERATION_MODE_ERROR,
*     - CE2_RSA_MGF_ILLEGAL_ARG_ERROR,
*     - CE2_RSA_PKCS1_VER_ARG_ERROR,
*     - CE2_RSA_INVALID_MESSAGE_DATA_SIZE,
*     - CE2_RSA_INVALID_OUTPUT_POINTER_ERROR,
*     - CE2_RSA_INVALID_SIGNATURE_BUFFER_SIZE.
*
* \brief \b
* Description:
*  RSA_Sign implements the Signing algorithm as defined
*  in PKCS#1 v1.5 or v2.1, using a single function
*  The actual macros that will be used by the user are:
*   CE2_RSA_PKCS1v15_Sign		- for v1.5;
*   CE2_RSA_PSS_Sign			- for v2.1.
*								- for v2.1 MD5 is not supported, since 
*                 according to the PKCS#1 ver2.1 it is not recommended.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DX_RSA_Sign
*     to sign data using private key, hash function and mask
*     generation functuion with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_RSA_Sign(CE2_RSAUserPrivKey_t *UserPrivKey_ptr,
                         CE2_RSA_HASH_OpMode_t hashFunc,
                         CE2_PKCS1_MGF_t MGF,
                         DxUint16_t SaltLen,				  			      
                         DxUint8_t     *DataIn_ptr,
                         DxUint32_t     DataInSize,
                         DxUint8_t     *Output_ptr,
                         DxUint16_t    *OutputSize_ptr,
                         CE2_PKCS1_version PKCS1_ver);

/**
****************************************************************
* Function Name: 
*  CE2_RSA_PKCS1v15_Sign
*
* Inputs:
*  see CE2_DX_RSA_Sign
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_PKCS1v15_Sign implements the RSASSA-PKCS1v15 algorithm 
*  as defined in PKCS#1 v1.5
*
*  This function combines the RSASP1 signature primitive and the
*  EMSA-PKCS1v15 encoding method, to provide an RSA-based signature scheme. 
*  For more details, please refer to the PKCS#1 standard.
*
* \b
* Algorithm:
*  -# call CE2_DX_RSA_Sign
***************************************************************/
#define CE2_RSA_PKCS1v15_Sign(UserPrivKey_ptr,hashFunc,DataIn_ptr,DataInSize,Output_ptr,OutputSize_ptr)\
		CE2_DX_RSA_Sign((UserPrivKey_ptr),(hashFunc),(CE2_PKCS1_NO_MGF),0,(DataIn_ptr),(DataInSize),(Output_ptr),(OutputSize_ptr),CE2_PKCS1_VER15)

/**
****************************************************************
* Function Name: 
*  CE2_RSA_PSS_Sign
*
* Inputs:
*  see CE2_DX_RSA_Sign
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_PSS_Sign implements the RSASSA-PSS algorithm
*  as defined in PKCS#1 v2.1 9.1, in a single function call.
*
*  Note: According to the PKCS#1 ver2.1 it is not recommended to use MD5 Hash, 
*        therefore it is not supported
*
*  The actual macro that will be used by the user is:
*  CE2_RSA_PSS_Sign
*
* \b
* Algorithm:
*  -# call CE2_DX_RSA_Sign
***************************************************************/
#define CE2_RSA_PSS_Sign(UserPrivKey_ptr,hashFunc,MGF,SaltLen,DataIn_ptr,DataInSize,Output_ptr,OutputSize_ptr)\
		CE2_DX_RSA_Sign(UserPrivKey_ptr,hashFunc,MGF,SaltLen,DataIn_ptr,DataInSize,Output_ptr,OutputSize_ptr,CE2_PKCS1_VER21)	

/**
****************************************************************
* Function Name: 
*  CE2_DX_RSA_Verify
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key data structure of the user;
*  @param hashFunc       [in] - The hash function to be used. 
*         Currently only CE2_PKCS1_SHA1 and CE2_PKCS1_MD5
*         are supported. The other hash functions
*         recommended by PKCS#1 v2.1 are
*         SHA-256/284/512.  If PKCS1_ver is CE2_PKCS1_VER15, then
*         the value CE2_RSA_After_HASH_NOT_KNOWN_mode will allow
*         the signature data to determine the hash function to be used.
*  @param MGF            [in] - The mask generation function. only for PKCS#1 v2.1
*         defines MGF1, so the only value allowed for v2.1 
*         is CE2_PKCS1_MGF1. 
*  @param SaltLen        [in] - The Length of the Salt buffer. relevant for PKCS#1 Ver 2.1 Only
*         Typical lengths are 0 and hLen (20 for SHA1)
*         The maximum length allowed is NSize - hLen - 2
*  @param DataIn_ptr     [in] - A pointer to the data whose signature is to be verified;
*  @param DataInSize     [in] - The size, in bytes, of the data whose signature is to be verified;
*  @param Sig_ptr        [in] - A pointer to the signature to be verified;
*         The length of the signature is PubKey_ptr->N.len bytes 
*         (that is, the size of the modulus, in bytes).
*  @param PKCS1_ver      [in] - Ver 1.5 or 2.1, according to the functionality required;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_INVALID_SIGNATURE,
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  RSA_Verify implements the RSASSA-PKCS1v15 algorithm
*  in a single function, as defined in PKCS#1 v1.5 and v2.1
*  The actual macro that will be used by the users is:
*   CE2_RSA_PKCS1v15_Verify			- for v1.5;
*   CE2_RSA_PSS_Verify				- for v2.1;
*									- for v2.1 MD5 is not supported, since
*					according to the PKCS#1 ver2.1 it is not recommended.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF__DX_RSA_Verify
*     to verify data using public key, hash function, mask
*     generation functuion and sign of data with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_RSA_Verify(CE2_RSAUserPubKey_t *UserPubKey_ptr,
                           CE2_RSA_HASH_OpMode_t hashFunc,
                           CE2_PKCS1_MGF_t MGF,
                           DxUint16_t SaltLen,
                           DxUint8_t     *DataIn_ptr,
                           DxUint32_t     DataInSize,
                           DxUint8_t     *Sig_ptr,
                           CE2_PKCS1_version PKCS1_ver);

/**
****************************************************************
* Function Name: 
*  CE2_RSA_PKCS1v15_Verify
*
* Inputs:
*  see CE2_DX_RSA_Verify
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  \brief CE2_RSA_PKCS1v15_Verify implements the PKCS1v15 Verify algorithm
*  as defined in PKCS#1 v1.5, with operating the HASH function
*   
* \b
* Algorithm:
*  -# call CE2_DX_RSA_Verify.
***************************************************************/
#define CE2_RSA_PKCS1v15_Verify(UserPubKey_ptr,hashFunc,DataIn_ptr,DataInSize,Sig_ptr)\
		CE2_DX_RSA_Verify(UserPubKey_ptr,hashFunc,CE2_PKCS1_NO_MGF,0,DataIn_ptr,DataInSize,Sig_ptr,CE2_PKCS1_VER15)

/**
****************************************************************
* Function Name: 
*  CE2_RSA_PSS_Verify
*
* Inputs:
*  see CE2_DX_RSA_Verify
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  \brief CE2_RSA_PSS_Verify implements the PKCS1v21 Verify algorithm
*  as defined in PKCS#1 v2.1, with operating the HASH function
*
* \b
* Algorithm:
*  -# call CE2_DX_RSA_Verify.
***************************************************************/
#define CE2_RSA_PSS_Verify(UserPubKey_ptr,hashFunc,MGF,SaltLen,DataIn_ptr,DataInSize,Sig_ptr)\
		CE2_DX_RSA_Verify(UserPubKey_ptr,hashFunc,MGF,SaltLen,DataIn_ptr,DataInSize,Sig_ptr,CE2_PKCS1_VER21)

/**
***************************************************************
* Function Name: 
*  CE2_RSA_PSS_SHA1_Verify
*
* Inputs:
*  see CE2_DX_RSA_Verify
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  \brief CE2_RSA_PSS_SHA1_Verify implements the PKCS1v21 Verify algorithm
*  as defined in PKCS#1 v2.1, but without operating the HASH function - 
*  it assumes the DataIn_ptr has already been hashed using SHA1
*   
*  \note The data_in size is already known after the Hash
*
*  The actual macro that will be used by the user is:
*  CE2_RSA_PKCS1v15_SHA1_Verify
*
* \b
* Algorithm:
*  -# call CE2_DX_RSA_Verify.
***************************************************************/
#define CE2_RSA_PSS_SHA1_Verify(UserPubKey_ptr,MGF,SaltLen,DataIn_ptr,Sig_ptr)\
		CE2_DX_RSA_Verify(UserPubKey_ptr,CE2_RSA_After_SHA1_mode,MGF,SaltLen,DataIn_ptr,CE2_HASH_SHA1_DIGEST_SIZE_IN_BYTES,Sig_ptr,CE2_PKCS1_VER21)

#ifdef __cplusplus
}
#endif			

#endif /* CE2_RSA_SCHEMES_H */
